﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    class BigPogobotEntity : EnemyEntity
    {
        enum eStatePobogot
        {
            eStateJump,
            eSateFall,
            eStateGround,
            eStateCount
        };

        /// <summary>
        /// Inner state of the pogobot
        /// </summary>
        eStatePobogot _eState;

        /// <summary>
        /// Texture's id for the sprite of the Pogobot on the ground
        /// </summary>
        int _iIdTextureGround;

        /// <summary>
        /// Sprite's id of the Pogobot on the ground
        /// </summary>
        int _iIdSpriteGround;

        /// <summary>
        /// Texture's id for the Pogobot jumping
        /// </summary>
        int _iIdTextureJump;

        /// <summary>
        /// Sprite's id of the Pogobot jumping
        /// </summary>
        int _iIdSpriteJump;

        /// <summary>
        /// The enemy shape
        /// </summary>
        //DynamicShapeRectangle _shape;

        /// <summary>
        /// Falg set to true when the state is changed
        /// </summary>
        bool _bPreState;

        /// <summary>
        /// Position when the enemy starts to jump
        /// </summary>
        Vector2 _v2StartPositionJump;

        /// <summary>
        /// Position when the enemy starts to fall
        /// </summary>
        Vector2 _v2StartPositionFall;

        /// <summary>
        /// Start time when the enemy landed on the ground
        /// </summary>
        int _iStartTimeGround;

        /// <summary>
        /// Side speed (on the x axis)
        /// </summary>
        const int SIDE_SPEED = 1;

        /// <summary>
        /// Max up and down speed (Y axis)
        /// </summary>
        const int UP_AND_DOWN_SPEED = 4;

        /// <summary>
        /// Total time to accelerate to full speed going up or down
        /// </summary>
        const int ACCELERATION_TIME = 1000;

        /// <summary>
        /// Current time of acceleration
        /// </summary>
        int _iCurrentAccelerationTime;

        /// <summary>
        /// Start time of acceleration
        /// </summary>
        int _iAccelerationStartTime;

        /// <summary>
        /// Displacement direction (1 : right, -1 : left)
        /// </summary>
        int _iDirection;

        /// <summary>
        /// Current side speed (precomputed when the enemy enter the jump state)
        /// </summary>
        int _iCurrentSideSpeed;

        /// <summary>
        /// Position on the x axis of the player
        /// </summary>
        int _iTarget;

        /// <summary>
        /// Animation Id of the explosion
        /// </summary>
        int _iAnimationExplosionID;

        /// <summary>
        /// Position offset for the explosion
        /// </summary>
        Vector2 _v2ExplosionOffset;

        /// <summary>
        /// Random number generator
        /// </summary>
        System.Random _rGen;


#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml";}}
        public static string EDITOR_SPRITE { get { return "bigPogobot_1"; } }
#endif
        /// <summary>
        /// Constructor
        /// </summary>
        public BigPogobotEntity() 
            :base()
        {
            _iIdTextureGround = -1;
            _iIdSpriteGround = -1;
            _iIdTextureJump = -1;
            _iIdSpriteJump = -1;
            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._bCollisionEnable = false;
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;
            _bPreState = true;
            _side = eSide.eSideEnemy;
            _rGen = null;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading.
        /// </summary>
        public override void init()
        {
            //get the graphics id
            _iAnimationExplosionID = Visu.Instance.getAnimationID("Little_Explosion");
            _iIdTextureGround = Visu.Instance.loadTilset("enemysheet.xml");
            _iIdTextureJump = _iIdTextureGround;
            _iIdSpriteGround = Visu.Instance.getSpriteId(_iIdTextureGround, "bigPogobot_1");
            _iIdSpriteJump = Visu.Instance.getSpriteId(_iIdTextureJump, "bigPogobot_2");

            //calculate the explosion offset
            int width = Visu.Instance.getSpriteWidth(_iIdTextureGround, _iIdSpriteGround);
            int height = Visu.Instance.getSpriteHeight(_iIdTextureGround, _iIdSpriteGround);
            _v2ExplosionOffset = new Vector2(width / 2, height/2);

            //create the shape
            _shape.resize(Visu.Instance.getSpriteWidth(_iIdTextureGround, _iIdSpriteGround), Visu.Instance.getSpriteHeight(_iIdTextureGround, _iIdSpriteGround));     
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            _entityType = eEntityType.eEntityEnemy;
            _rGen = new System.Random(TimeClock.Clock.instance.millisecs);
            base.init();
        }

        /// <summary>
        /// Activate the entity
        /// </summary>
        /// <param name="position">world position</param>
        //public void activate(int idTextureGround, int idSpriteGround, int idTextureJump, int idSpriteJump, Vector2 position)
        //public void activate(Vector2 position)
        public override void activate()
        {
            _eState = eStatePobogot.eSateFall;

            _shape._bCollisionEnable = true;

            //_v2Position = position;
            _shape._v2position = Position;

            base.activate();
        }

        /// <summary>
        /// Update the entity
        /// </summary>
        public override void update()
        {
            
            switch (_eState)
            {
                case eStatePobogot.eStateGround:
                    updateStateGround();
                    break;
                case eStatePobogot.eSateFall:
                    updateStateFall();
                    break;
                case eStatePobogot.eStateJump:
                    updateStateJump();
                    break;
            }

            //check if collision with the player
            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null)
            {
                World.Instance.PlayerEntity.hurt(_iDamages);
            }
        }

        public void updateStateGround()
        {
            if (_bPreState)
            {
                _iStartTimeGround = TimeClock.Clock.instance.millisecs;
                _bPreState = false;
            }

            const int TotalTimeGround = 1000;
            if (TimeClock.Clock.instance.millisecs >= _iStartTimeGround + TotalTimeGround)
            {
                _eState = eStatePobogot.eStateJump;
                _bPreState = true;
            }
        }

        public void updateStateJump()
        {
            //initialise the variables for the jump state
            if (_bPreState)
            {
                _iAccelerationStartTime = TimeClock.Clock.instance.millisecs;
                _v2StartPositionJump = _v2Position;
                _bPreState = false;
                Vector2 direction = World.Instance.PlayerPosition - Position;
                _iDirection = direction.X > 0 ? 1 : -1;
                _iCurrentSideSpeed = SIDE_SPEED * _iDirection;
                _iTarget = (int)World.Instance.PlayerPosition.X;

                //to optimize, create the generator once
                //int iRandom = _rGen.Next(4);
                double iRandom = _rGen.NextDouble();
                _iCurrentSideSpeed += (int)(iRandom * _iDirection * 4);
            }
            
            //calculate the jump speed
            _iCurrentAccelerationTime = TimeClock.Clock.instance.millisecs - _iAccelerationStartTime;
            float elapsedTime = ACCELERATION_TIME - _iCurrentAccelerationTime;
            float ratio = elapsedTime / ACCELERATION_TIME;
            float fCurrentJumpSpeed = -UP_AND_DOWN_SPEED * ratio;
            if (_iCurrentAccelerationTime > ACCELERATION_TIME)
            {
                _eState = eStatePobogot.eSateFall;
                _bPreState = true;
                return;
            }
           
            //apply the displacement
            Vector2 displacement = new Vector2(_iCurrentSideSpeed, fCurrentJumpSpeed);
            _v2Position += displacement;
            _shape._v2position = _v2Position;

            //check for collision
            CollisionResult res = Physics.Physics.Instance.checkAllRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlatform);
            if (res != null) 
                _v2Position += res.Overlap;
        }

        public void updateStateFall()
        {
            if (_bPreState)
            {
                _iAccelerationStartTime = TimeClock.Clock.instance.millisecs;
                _v2StartPositionFall = Position;
                _bPreState = false;
            }

            //calculate the fall speed
            _iCurrentAccelerationTime = TimeClock.Clock.instance.millisecs - _iAccelerationStartTime;
            float ratio = (float)_iCurrentAccelerationTime / (float)ACCELERATION_TIME;
            float fCurrentJumpSpeed = UP_AND_DOWN_SPEED * ratio;

            //apply the displacement
            Vector2 displacement = new Vector2(_iCurrentSideSpeed, fCurrentJumpSpeed);
            _v2Position += displacement;
            _shape._v2position = _v2Position;

            //check collision
            CollisionResult res = Physics.Physics.Instance.checkAllRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlatform);
            if (res == null) return;

            //apply overlap
            _v2Position += res.Overlap;

            //if collision with the ground, go to ground state
            if (res.Overlap.Y < 0)
            {
                _eState = eStatePobogot.eStateGround;
                _bPreState = true;
            }
        }

        public override void render()
        {
            Microsoft.Xna.Framework.Graphics.SpriteEffects flip = Microsoft.Xna.Framework.Graphics.SpriteEffects.None;
            if (_iDirection == 1)
                flip = Microsoft.Xna.Framework.Graphics.SpriteEffects.FlipHorizontally;

            switch (_eState)
            {
                case eStatePobogot.eStateGround:
                    Visu.Instance.displaySprite(_iIdTextureGround, _iIdSpriteGround, ScreenPosition, flip);
                    break;
                case eStatePobogot.eSateFall:
                    Visu.Instance.displaySprite(_iIdTextureJump, _iIdSpriteJump, ScreenPosition, flip);
                    break;
                case eStatePobogot.eStateJump:
                    Visu.Instance.displaySprite(_iIdTextureJump, _iIdSpriteJump, ScreenPosition, flip);
                    break;
            }

#if DEBUG
            Vector2[] obb = _shape.getOrientedBoundingBox();
            for (int i = 0; i < 4; i++)
                obb[i] -= World.Instance.CameraPosition;

            Visu.Instance.displayPolygon(obb);
#endif
        }

        /// <summary>
        /// The entity die
        /// </summary>
        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iAnimationExplosionID, _v2Position + _v2ExplosionOffset);
        }

        //public override void hurt(int damages)
        //{
        //    HP -= damages;
        //    if (HP <= 0) die();
        //}
    }
}
